# Docs Plugin

[![GitHub Workflow Status (branch)](https://img.shields.io/github/workflow/status/wintercms/wn-docs-plugin/Tests/main?label=tests&style=flat-square)](https://github.com/wintercms/wn-docs-plugin/actions)
[![Codecov](https://img.shields.io/codecov/c/github/wintercms/wn-docs-plugin?style=flat-square)](https://codecov.io/gh/wintercms/wn-docs-plugin)
[![Discord](https://img.shields.io/discord/816852513684193281?label=discord&style=flat-square)](https://discord.gg/D5MFSPH6Ux)

Integrates a full suite of documentation direct into your Winter CMS installation. Documentation can be generated from Markdown files or analysed from PHP code.

## Features

- Generate documentation locally from your plugin, or from a remote ZIP file.
- Create content-based documentation from Markdown files.
- Create API documentation from PHP doc-blocks and parsed PHP code.
- Can be used in both the Backend and CMS.

## Getting started

To install the plugin, you may install it through the [Winter CMS Marketplace](https://github.com/wintercms/wn-docs-plugin), or you may install it using Composer:

```bash
composer require winter/wn-docs-plugin
```

Then, run the migrations to ensure the plugin is enabled:

```bash
php artisan winter:up
```

## Registering documentation

Documentation can be registered by adding a `registerDocumentation` method to your Plugin class (`Plugin.php`), and will depend on whether the documentation is content-based or API-based, and whether the documentation or code is stored locally or remotely.

```php
<?php

class MyPlugin extends \System\Classes\PluginBase
{
    // ...

    public function registerDocumentation()
    {
        return [
            'guide' => [
                'name' => 'Documentation Guide',
                'type' => 'user',
                'source' => 'local',
                'path' => 'docs'
            ],
        ];
    }

    // ...
}
```

The method should return an array, with the key of each item representing the "code" of the documentation, and the following parameters in an array as the value:

Parameter | Required | Description
--------- | -------- | -----------
`name` | Yes | The name of this documentation. This will be displayed in documentation lists.
`type` | Yes | The type of documentation. It must be one of the following: `md` or `php`. See the [Types of Documentation](#documentation-types) for more information.
`source` | Yes | Where the documentation can be sourced from. Must be either `local` or `remote`. See the [Documentation Sources](#documentation-sources) section for more information.
`path` | No | This will determine the path - relative to the plugin root if source is `local` or relative to the extracted content if `remote` - that the documentation or code can be found.
`url` | No | If `source` is remote, this will determine the URL to download the documentation source from. The URL must point to a ZIP file that can be extracted. Not needed if the `source` is `local`.
`zipFolder` | No | If `source` is remote, this will allow you to limit the source to a folder within the ZIP file, if the ZIP includes other files. You may also specify `true` to extract a single folder, if the folder name may be autogenerated. Ignored if the `source` is `local`.
`token` | No | If the `source` is remote, and is a private resource, you can add an authorization token to be sent with the HTTP request. Ignored if the source is `local`.
`tocPath` | No | Determines the path, relative to the source, where the table of contents YAML file can be found. By default, the Docs plugin will look for a `toc.yaml` in the root folder of the documentation source.
`image` | No | Provides an image representation of the documentation.
`ignorePaths` | No | An array of paths to ignore when finding available documentation. Each path may be specified as a glob.
`repository` | No | An array that contains the URL to the source repository, which allows the documentation to link back to the source, for example, to provide "Edit this page" links. You can specify both a `url` value, which represents the readable source location for both `md` and `php` type documentation, and an `editUrl` which represents the URL to edit the source of an `md` type documentation.

For API documentation (ie. the `type` parameter is `php`), there are a couple of extra parameters that may be specified:

Parameter | Description
--------- | -----------
`sourcePaths` | An array of paths to limit the API parser to. Each path may be specified as a glob. If no source paths are provided, all PHP files are parsed. Note that the `ignorePaths` patterns are still applied.

## Types of Documentation

The Docs plugin currently supports two types of documentation, Markdown (`md`) and PHP (`php`).

### Markdown

Markdown is used for the generation of textual and image-based documentation, allowing for the easy writing of large amounts of documentation in a short amount of time. The Markdown documentation processor uses the [CommonMark library](https://github.com/thephpleague/commonmark) to ensure accurate parsing of Markdown and the enabling of these useful features:

- Auto-linking of web and email addresses
- Automatic table of contents generation and anchor tags
- External link handling
- Front matter (title and meta definition)
- Markdown tables

Markdown documentation can be arranged in any way you see fit - the main table of contents can be specific in a `.yaml` file available within the documentation source.

#### Example table of contents YAML file

```yaml
rootPage: home
sections:
    Introduction:
        pages:
            welcome: "Welcome"

    Functionality:
        pages:
            functionality/cool-stuff: "Cool Stuff"
            functionality/big-things: "Big Things"
```

#### Example Markdown document

```md
---
title: Big Things
---

# Big Things

We have some big things available in this software.

## Awesome Feature One

This feature allows you to reach a new level of awesome.
```

### PHP

PHP documentation involves parsing a PHP source code directory and determining the available API within all the source code objects, such as classes, namespaces, properties and methods.

Using the awesome [PHP Parser library](https://github.com/nikic/php-parser), the code is analyzed and derived from the signatures of all aspects of a class, as well as doc-blocks written to provide additional context for those signatures.

This allows, for example, the following class:

```php
<?php

namespace Acme\Blog;

use MarkdownParser;

class Post extends \BasePost
{
    public string $title;

    public string $content;

    /**
     * Whether the post is published or not
     */
    protected boolean $published = false;

    /**
     * Renders the post and returns an array
     *
     * @return array Title and content
     */
    public function render()
    {
        return [
            'title' => $this->title,
            'content' => MarkdownParser::parse($this->content),
        ];
    }
}
```

To be rendered into a readable documentation that outlines the available properties (`$title`, `$content`) and method (`render()`) as readable documentation.
